/**
 * Professional IP Information Checker
 * Advanced JavaScript Module with Comprehensive API Integration
 * Author: Professional Web Developer
 * Version: 2.0.0
 */

class IPChecker {
    constructor() {
        this.currentData = {};
        this.map = null;
        this.apiResults = {};
        this.dualStack = null;
        // Aggregates for summary card
        this.aggregates = {
            cities: new Set(),
            regions: new Set(),
            countries: new Set(), // e.g., "Indonesia (ID)"
            isps: new Set(),
            asns: new Set(),
            zips: new Set()
        };
        // History removed for public version to avoid storing data
        
        // Enhanced API Configuration dengan banyak sumber (HTTPS only)
        this.apis = {
            ipinfo: { name: 'IPInfo.io', url: (ip) => `https://ipinfo.io/${ip}/json`, parser: 'parseIPInfo', priority: 1, rateLimited: false },
            ipapi_co: { name: 'IPAPI.co', url: (ip) => `https://ipapi.co/${ip}/json/`, parser: 'parseIPAPICo', priority: 2, rateLimited: false },
            ipwho_is: { name: 'IPWho.is', url: (ip) => `https://ipwho.is/${ip}`, parser: 'parseIPWhoIs', priority: 3, rateLimited: false },
            ipwhois_app: { name: 'IPWhois.app', url: (ip) => `https://ipwhois.app/json/${ip}`, parser: 'parseIPWhoisApp', priority: 4, rateLimited: false },
            geojs: { name: 'GeoJS', url: (ip) => `https://get.geojs.io/v1/ip/geo/${ip}.json`, parser: 'parseGeoJS', priority: 5, rateLimited: false },
            ipsb: { name: 'IP.sb', url: (ip) => `https://api.ip.sb/geoip/${ip}`, parser: 'parseIPSB', priority: 6, rateLimited: false },
            freegeoip: { name: 'FreeGeoIP.app', url: (ip) => `https://freegeoip.app/json/${ip}`, parser: 'parseFreeGeoIP', priority: 7, rateLimited: false },
            dbip: { name: 'DB-IP Free', url: (ip) => `https://api.db-ip.com/v2/free/${ip}`, parser: 'parseDBIP', priority: 8, rateLimited: false },
            iplocate: { name: 'IPLocate.io', url: (ip) => `https://www.iplocate.io/api/lookup/${ip}`, parser: 'parseIPLocate', priority: 9, rateLimited: false },
            geolocation_db: { name: 'Geolocation-DB', url: (ip) => `https://geolocation-db.com/json/${ip}`, parser: 'parseGeolocationDB', priority: 10, rateLimited: false },
            seeip: { name: 'seeip.org', url: (ip) => `https://ip.seeip.org/geoip/${ip}`, parser: 'parseSeeip', priority: 11, rateLimited: false },
            freeipapi: { name: 'FreeIPAPI', url: (ip) => `https://freeipapi.com/api/json/${ip}`, parser: 'parseFreeIPAPI', priority: 12, rateLimited: false },
            ipapi_is: { name: 'ipapi.is', url: (ip) => `https://ipapi.is/?q=${ip}`, parser: 'parseIPAPIIs', priority: 13, rateLimited: false },
            techniknews: { name: 'TechnikNews IPGeo', url: (ip) => `https://api.techniknews.net/ipgeo/${ip}`, parser: 'parseTechnikNews', priority: 14, rateLimited: false },
            iplocation_net: { name: 'IPLocation.net', url: (ip) => `https://api.iplocation.net/?ip=${ip}`, parser: 'parseIPLocationNet', priority: 15, rateLimited: false },
            geoiplookup: { name: 'GeoIPLookup.io', url: (ip) => `https://json.geoiplookup.io/${ip}`, parser: 'parseGeoIPLookup', priority: 16, rateLimited: false },
            sypexgeo: { name: 'SypexGeo', url: (ip) => `https://api.sypexgeo.net/json/${ip}`, parser: 'parseSypexGeo', priority: 17, rateLimited: false },
            geoplugin: { name: 'geoPlugin', url: (ip) => `https://www.geoplugin.net/json.gp?ip=${ip}`, parser: 'parseGeoPlugin', priority: 18, rateLimited: false }
        };

        // Privacy/Security Detection Patterns
        this.privacyPatterns = {
            vpnProviders: [
                'ExpressVPN', 'NordVPN', 'Surfshark', 'CyberGhost', 'Private Internet Access',
                'ProtonVPN', 'IPVanish', 'VyprVPN', 'TunnelBear', 'Windscribe',
                'Perfect Privacy', 'Hide.me', 'VPN Unlimited', 'TorGuard', 'Mullvad',
                'PureVPN', 'StrongVPN', 'SaferVPN', 'ZenMate', 'VPN.ac'
            ],
            cloudProviders: [
                'Amazon', 'Google', 'Microsoft', 'DigitalOcean', 'Vultr', 'Linode',
                'OVH', 'Hetzner', 'Scaleway', 'Cloudflare', 'Fastly', 'KeyCDN'
            ],
            torExitNodes: [],
            datacenters: [
                'Datacenter', 'Data Center', 'Hosting', 'Server', 'Cloud', 'CDN',
                'Content Delivery', 'Colo', 'Colocation'
            ]
        };

        this.init();
    }

    init() {
        this.setupEventListeners();
        this.detectMyIP();
        this.updatePrivacyScore();
        this.fetchDualStackIPs();
        this.initCardToggles(); // Initialize toggle buttons on page load
    }

    setupEventListeners() {
        // Input events
        const ipInput = document.getElementById('ipInput');
        if (ipInput) {
            ipInput.addEventListener('keypress', (e) => {
                if (e.key === 'Enter') this.checkIP();
            });
            
            ipInput.addEventListener('input', (e) => {
                // Remove all spaces automatically
                const cleanValue = e.target.value.replace(/\s/g, '');
                if (e.target.value !== cleanValue) {
                    e.target.value = cleanValue;
                }
                this.validateInputRealtime(cleanValue);
            });
            
            // Also handle paste events to clean spaces immediately
            ipInput.addEventListener('paste', (e) => {
                setTimeout(() => {
                    const cleanValue = e.target.value.replace(/\s/g, '');
                    if (e.target.value !== cleanValue) {
                        e.target.value = cleanValue;
                    }
                    this.validateInputRealtime(cleanValue);
                }, 0);
            });
        }

        // Button events
        const checkBtn = document.getElementById('checkBtn');
        if (checkBtn) checkBtn.addEventListener('click', () => this.checkIP());

        // Dual-stack self detector button (no input)
        const convertBtn = document.getElementById('convertBtn');
        if (convertBtn) {
            convertBtn.addEventListener('click', () => this.handleDualStackSelf());
        }

        // Collapsible toggles removed per new UX: cards always open
    }

    async fetchDualStackIPs() {
        try {
            this.dualStack = await IPUtils.getDualStackIPs();
        } catch (e) {
            this.dualStack = { ipv4: null, ipv6: null };
        }
    }

    async detectMyIP() {
        const myIpv4Element = document.getElementById('myIpv4');
        const myIpv6Element = document.getElementById('myIpv6');
        const myLocationElement = document.getElementById('myLocation');
        
        if (!myIpv4Element || !myIpv6Element || !myLocationElement) return;

        // Set initial detecting state
        myIpv4Element.textContent = 'Detecting...';
        myIpv6Element.textContent = 'Detecting...';
        myLocationElement.textContent = 'Detecting...';

        try {
            // Detect both IPv4 and IPv6 simultaneously
            const dualStackResult = await IPUtils.getDualStackIPs();
            
            let locationIP = null;
            
            // Update IPv4 - validate it's actually IPv4
            if (dualStackResult.ipv4) {
                const ipv4Validation = IPUtils.validateIP(dualStackResult.ipv4);
                if (ipv4Validation.valid && ipv4Validation.type === 'IPv4') {
                    myIpv4Element.textContent = dualStackResult.ipv4;
                    this.updateCurrentIPInfo(dualStackResult.ipv4);
                    locationIP = dualStackResult.ipv4;
                } else {
                    myIpv4Element.textContent = 'Not detected';
                }
            } else {
                myIpv4Element.textContent = 'Not detected';
            }
            
            // Update IPv6 - validate it's actually IPv6
            if (dualStackResult.ipv6) {
                const ipv6Validation = IPUtils.validateIP(dualStackResult.ipv6);
                if (ipv6Validation.valid && ipv6Validation.type === 'IPv6') {
                    myIpv6Element.textContent = dualStackResult.ipv6;
                    // If no IPv4 was detected, use IPv6 for location
                    if (!locationIP) {
                        locationIP = dualStackResult.ipv6;
                    }
                } else {
                    myIpv6Element.textContent = 'Not detected';
                }
            } else {
                myIpv6Element.textContent = 'Not detected';
            }
            
            // Get location info using the available IP (prefer IPv4, then IPv6)
            if (locationIP) {
                this.getLocationForIP(locationIP, myLocationElement);
            } else {
                // If neither IPv4 nor IPv6 were detected from dual-stack, try fallback
                await this.fallbackIPDetection(myIpv4Element, myLocationElement);
                // Keep IPv6 as "Not detected" since fallback only provides IPv4
                myIpv6Element.textContent = 'Not detected';
            }
            
        } catch (error) {
            console.error('Error detecting dual-stack IPs:', error);
            // Fallback to original detection method (IPv4 only)
            await this.fallbackIPDetection(myIpv4Element, myLocationElement);
            myIpv6Element.textContent = 'Not detected';
        }
    }

    async fallbackIPDetection(ipv4Element, locationElement) {
        try {
            // Try multiple IP detection services (these typically return IPv4)
            const ipServices = [
                'https://api.ipify.org?format=json',
                'https://ipinfo.io/ip',
                'https://api.myip.com',
                'https://httpbin.org/ip'
            ];

            for (const service of ipServices) {
                try {
                    const response = await fetch(service);
                    const data = await response.text();
                    
                    let ip;
                    try {
                        const jsonData = JSON.parse(data);
                        ip = jsonData.ip || jsonData.origin;
                    } catch {
                        ip = data.trim();
                    }

                    if (ip) {
                        const validation = IPUtils.validateIP(ip);
                        if (validation.valid && validation.type === 'IPv4') {
                            ipv4Element.textContent = ip;
                            this.updateCurrentIPInfo(ip);
                            this.getLocationForIP(ip, locationElement);
                            return;
                        }
                    }
                } catch (error) {
                    console.warn(`Failed to get IP from ${service}:`, error);
                }
            }
            
            ipv4Element.textContent = 'Not detected';
            locationElement.textContent = 'Not detected';
        } catch (error) {
            console.error('Error in fallback detection:', error);
            ipv4Element.textContent = 'Error';
            locationElement.textContent = 'Error';
        }
    }

    async getLocationForIP(ip, locationElement) {
        try {
            // Use a quick geolocation service to get basic location info
            const response = await fetch(`https://ipinfo.io/${ip}/json`);
            const data = await response.json();
            
            if (data.city && data.country) {
                locationElement.textContent = `${data.city}, ${data.country}`;
            } else if (data.country) {
                locationElement.textContent = data.country;
            } else {
                locationElement.textContent = 'Location unknown';
            }
        } catch (error) {
            console.warn('Failed to get location:', error);
            locationElement.textContent = 'Location unknown';
        }
    }

    async handleDualStackSelf() {
        const resultBox = document.getElementById('converterResult');
        if (!resultBox) return;

        // Show loading
        resultBox.innerHTML = `
            <div class="loading-box">
                <div class="spinner"></div>
                <p class="loading-text">Detecting your IPv4/IPv6...</p>
            </div>
        `;

        try {
            // Parallel detection of client's own IPv4 and IPv6
            const [{ ipv4, ipv6 }] = await Promise.all([
                IPUtils.getDualStackIPs()
            ]);

            const lines = [];
            lines.push(`<div class="info-item"><span class="info-label">Detection Scope</span><span class="info-value">Your current connection</span></div>`);

            // IPv4 result
            if (ipv4) {
                lines.push(`<div class="info-item"><span class="info-label">IPv4</span><span class="info-value font-mono" style="color: var(--accent-green);">${ipv4}</span></div>`);
            } else {
                lines.push(`<div class="info-item"><span class="info-label">IPv4</span><span class="info-value">Not detected</span></div>`);
            }

            // IPv6 result
            if (ipv6) {
                lines.push(`<div class="info-item"><span class="info-label">IPv6</span><span class="info-value font-mono" style="color: var(--accent-green);">${ipv6}</span></div>`);
            } else {
                lines.push(`<div class="info-item"><span class="info-label">IPv6</span><span class="info-value">Not detected</span></div>`);
            }

            // Derived info
            if (ipv6) {
                const conv = IPUtils.ipv6ToIpv4(ipv6);
                if (conv && conv.success && conv.ipv4) {
                    lines.push(`<div class="info-item"><span class="info-label">IPv4-mapped (from IPv6)</span><span class="info-value font-mono">${conv.ipv4}</span></div>`);
                    lines.push(`<div class="info-item"><span class="info-label">Mapped Method</span><span class="info-value">${conv.method}</span></div>`);
                }
            }

            // Methods and disclaimer per reference
            lines.push(`<div class="info-item"><span class="info-label">Methods</span><span class="info-value">IPv4-mapped extraction; Parallel IPv4/IPv6 detection</span></div>`);
            lines.push(`<div class="info-item"><span class="info-label">Disclaimer</span><span class="info-value">Dual-stack depends on your current network. This detects your connection only.</span></div>`);

            resultBox.innerHTML = lines.join('');
        } catch (error) {
            console.error('Dual-stack self detection error:', error);
            resultBox.innerHTML = `<div class="info-item"><span class="info-label">Error</span><span class="info-value">Failed to detect your IPv4/IPv6.</span></div>`;
        }
    }

    async updateCurrentIPInfo(ip) {
        try {
            // Quick check for current IP basic info
            const response = await fetch(`https://ipapi.co/${ip}/json/`);
            const data = await response.json();
            
            if (data && data.country_name) {
                const currentIpDiv = document.getElementById('currentIp');
                if (currentIpDiv) {
                    const isIPv6 = ip.includes(':');
                    const versionLabel = isIPv6 ? 'IPv6' : 'IPv4';
                    currentIpDiv.innerHTML = `
                        <div class="current-ip-row">
                            <div class="current-ip-icon"><i class="fas fa-wifi"></i></div>
                            <div class="current-ip-main">
                                <div class="current-ip-value"><strong>${versionLabel} | ${ip}</strong></div>
                            </div>
                        </div>
                        <div class="current-ip-location">
                            <i class="fas fa-map-marker-alt"></i>
                            <span>${data.city || 'Unknown'}, ${data.country_name || 'Unknown'}</span>
                        </div>`;
                }
            }
        } catch (error) {
            console.warn('Could not get current IP info:', error);
        }
    }

    validateIP(ip) {
        if (!ip) return false;
        
        // IPv4 regex
        const ipv4Regex = /^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/;
        
        // IPv6 regex (simplified)
        const ipv6Regex = /^(?:[0-9a-fA-F]{1,4}:){7}[0-9a-fA-F]{1,4}$|^::1$|^::$|^(?:[0-9a-fA-F]{1,4}:)*::(?:[0-9a-fA-F]{1,4}:)*[0-9a-fA-F]{1,4}$/;
        
        return ipv4Regex.test(ip) || ipv6Regex.test(ip);
    }

    validateInputRealtime(ip) {
        const ipInput = document.getElementById('ipInput');
        const checkBtn = document.getElementById('checkBtn');
        
        if (!ipInput || !checkBtn) return;

        if (ip && this.validateIP(ip)) {
            ipInput.classList.remove('invalid');
            ipInput.classList.add('valid');
            checkBtn.disabled = false;
        } else if (ip) {
            ipInput.classList.add('invalid');
            ipInput.classList.remove('valid');
            checkBtn.disabled = true;
        } else {
            ipInput.classList.remove('invalid', 'valid');
            checkBtn.disabled = false;
        }
    }

    async checkIP() {
        const ipInput = document.getElementById('ipInput').value.trim();
        
        if (!ipInput) {
            this.showError('Please enter a valid IP address');
            return;
        }

        if (!this.validateIP(ipInput)) {
            this.showError('Invalid IP address format. Use IPv4 (e.g., 192.168.1.1) or IPv6');
            return;
        }

        this.showLoading();
        this.clearError();
        
        // Reset data
        this.currentData = { ip: ipInput };
        this.apiResults = {};
        this.aggregates = {
            cities: new Set(),
            regions: new Set(),
            countries: new Set(),
            isps: new Set(),
            asns: new Set(),
            zips: new Set()
        };
        
        // Start progress tracking
        this.updateProgress(0);
        
        // Get API calls with timeout and retry logic
        const apiCalls = Object.entries(this.apis).map(([key, api]) => 
            this.callAPIWithRetry(key, api, ipInput)
        );

    // Execute API calls with limited concurrency to avoid overloading
    const results = await this.runWithConcurrency(apiCalls, 3);
        
        this.updateProgress(100);
        this.hideLoading();
        
        // Process results
        this.processAPIResults(results);
        
        if (Object.keys(this.currentData).length <= 1) {
            this.showError('Could not fetch IP information from any source. Please try again.');
            return;
        }

        // Display results
        this.displayResults();
        this.showAPIStatus();
        this.calculatePrivacyScore();
    }

    async runWithConcurrency(promises, limit = 3) {
        const results = new Array(promises.length);
        let i = 0;
        let active = 0;

        return new Promise((resolve) => {
            const next = () => {
                while (active < limit && i < promises.length) {
                    const currentIndex = i++;
                    active++;
                    Promise.resolve(promises[currentIndex])
                        .then((res) => { results[currentIndex] = { status: 'fulfilled', value: res }; })
                        .catch((err) => { results[currentIndex] = { status: 'rejected', reason: err }; })
                        .finally(() => {
                            active--;
                            if (results.filter(Boolean).length === promises.length) {
                                resolve(results);
                            } else {
                                next();
                            }
                        });
                }
            };
            next();
        });
    }

    async callAPIWithRetry(apiKey, api, ip, retries = 1) {
        const startTime = Date.now();
        
        for (let attempt = 0; attempt <= retries; attempt++) {
            try {
                const controller = new AbortController();
                const timeoutId = setTimeout(() => controller.abort(), 5000); // 5s timeout
                
                const options = {
                    method: 'GET',
                    signal: controller.signal
                };
                
                if (api.headers) {
                    options.headers = api.headers;
                }
                
                const response = await fetch(api.url(ip), options);
                clearTimeout(timeoutId);
                
                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}`);
                }
                
                const data = await response.json();
                const responseTime = Date.now() - startTime;
                
                this.apiResults[apiKey] = {
                    status: 'success',
                    data: data,
                    responseTime: responseTime,
                    attempt: attempt + 1
                };
                
                // Parse the data
                if (this[api.parser]) {
                    this[api.parser](data);
                }
                
                return { apiKey, status: 'success', data, responseTime };
                
            } catch (error) {
                if (attempt === retries) {
                    const responseTime = Date.now() - startTime;
                    this.apiResults[apiKey] = {
                        status: 'error',
                        error: error.message,
                        responseTime: responseTime,
                        attempt: attempt + 1
                    };
                    return { apiKey, status: 'error', error: error.message };
                }
                
                // Wait before retry (exponential backoff light)
                await new Promise(resolve => setTimeout(resolve, 500 * (attempt + 1)));
            }
        }
    }

    // Enhanced API Parsers with comprehensive data extraction
    // IP-API removed from runtime due to HTTP-only on free tier

    parseIPInfo(data) {
        if (data.ip) {
            const parsedData = {
                hostname: data.hostname,
                city: data.city,
                region: data.region,
                country: data.country,
                org: data.org,
                postal: data.postal,
                timezone: data.timezone
            };
            
            if (data.loc) {
                const [lat, lon] = data.loc.split(',');
                parsedData.lat = parseFloat(lat);
                parsedData.lon = parseFloat(lon);
            }
            
            this.mergeData(parsedData);
        }
    }

    parseIPWhoIs(data) {
        if (data && (data.ip || data.success)) {
            this.mergeData({
                ip: data.ip,
                type: data.type,
                continent: data.continent,
                country: data.country,
                countryCode: data.country_code,
                region: data.region,
                city: data.city,
                zip: data.postal,
                lat: data.latitude,
                lon: data.longitude,
                timezone: data.timezone,
                isp: data.connection?.isp || data.connection?.org || data.isp,
                org: data.connection?.org || data.org,
                asn: data.connection?.asn || data.asn,
                proxy: data.security?.proxy,
                hosting: data.security?.hosting
            });
        }
    }

    parseIPWhoisApp(data) {
        if (data && data.ip) {
            this.mergeData({
                ip: data.ip,
                country: data.country,
                countryCode: data.country_code,
                region: data.region,
                city: data.city,
                zip: data.postal,
                lat: parseFloat(data.latitude),
                lon: parseFloat(data.longitude),
                timezone: data.timezone,
                isp: data.isp,
                org: data.org,
                as: data.as,
                proxy: data.proxy,
                hosting: data.hosting
            });
        }
    }

    parseGeoJS(data) {
        if (data && (data.ip || data.latitude)) {
            this.mergeData({
                ip: data.ip,
                country: data.country,
                countryCode: data.country_code,
                region: data.region,
                city: data.city,
                lat: parseFloat(data.latitude),
                lon: parseFloat(data.longitude),
                timezone: data.time_zone,
                org: data.organization,
                asn: data.asn
            });
        }
    }

    parseIPSB(data) {
        if (data && (data.ip || data.latitude)) {
            this.mergeData({
                ip: data.ip,
                country: data.country,
                countryCode: data.country_code,
                region: data.region,
                city: data.city,
                lat: parseFloat(data.latitude),
                lon: parseFloat(data.longitude),
                isp: data.isp,
                org: data.org,
                asn: data.asn
            });
        }
    }

    parseFreeGeoIP(data) {
        if (data && data.ip) {
            this.mergeData({
                ip: data.ip,
                country: data.country_name,
                countryCode: data.country_code,
                region: data.region_name,
                city: data.city,
                zip: data.zip_code,
                lat: parseFloat(data.latitude),
                lon: parseFloat(data.longitude),
                timezone: data.time_zone
            });
        }
    }

    parseDBIP(data) {
        if (data && (data.ipAddress || data.latitude)) {
            this.mergeData({
                ip: data.ipAddress,
                continent: data.continentName,
                country: data.countryName,
                countryCode: data.countryCode,
                region: data.stateProv,
                city: data.city,
                lat: parseFloat(data.latitude),
                lon: parseFloat(data.longitude)
            });
        }
    }

    parseIPLocate(data) {
        if (data && (data.ip || data.latitude)) {
            this.mergeData({
                ip: data.ip,
                country: data.country,
                countryCode: data.country_code,
                continent: data.continent,
                region: data.subdivision || data.region,
                city: data.city,
                lat: parseFloat(data.latitude),
                lon: parseFloat(data.longitude),
                timezone: data.timezone,
                org: data.organization,
                asn: data.asn
            });
        }
    }

    parseGeolocationDB(data) {
        if (data && (data.IPv4 || data.latitude)) {
            this.mergeData({
                ip: data.IPv4,
                country: data.country_name,
                countryCode: data.country_code,
                region: data.state,
                city: data.city,
                zip: data.postal,
                lat: parseFloat(data.latitude),
                lon: parseFloat(data.longitude)
            });
        }
    }

    parseSeeip(data) {
        if (data && (data.ip || data.latitude)) {
            this.mergeData({
                ip: data.ip,
                country: data.country,
                countryCode: data.country_code,
                region: data.region,
                city: data.city,
                lat: parseFloat(data.latitude),
                lon: parseFloat(data.longitude),
                org: data.organization,
                asn: data.asn
            });
        }
    }

    parseIPAPICo(data) {
        if (data.ip) {
            this.mergeData({
                country: data.country_name,
                countryCode: data.country_code,
                region: data.region,
                city: data.city,
                zip: data.postal,
                lat: data.latitude,
                lon: data.longitude,
                timezone: data.timezone,
                isp: data.org,
                currency: data.currency,
                languages: data.languages,
                asn: data.asn,
                calling_code: data.country_calling_code,
                country_tld: data.country_tld
            });
        }
    }

    parseIPGeolocation(data) {
        if (data.ip) {
            this.mergeData({
                country: data.country_name,
                countryCode: data.country_code2,
                region: data.state_prov,
                city: data.city,
                zip: data.zipcode,
                lat: parseFloat(data.latitude),
                lon: parseFloat(data.longitude),
                timezone: data.time_zone?.name,
                isp: data.isp,
                currency: data.currency?.code,
                continent: data.continent_name
            });
        }
    }

    parseIPData(data) {
        if (data.ip) {
            this.mergeData({
                country: data.country_name,
                countryCode: data.country_code,
                region: data.region,
                city: data.city,
                zip: data.postal,
                lat: data.latitude,
                lon: data.longitude,
                timezone: data.time_zone?.name,
                isp: data.organisation,
                asn: data.asn?.asn,
                threat: data.threat
            });
        }
    }

    parseIPRegistry(data) {
        if (data.ip) {
            this.mergeData({
                country: data.location?.country?.name,
                countryCode: data.location?.country?.code,
                region: data.location?.region?.name,
                city: data.location?.city,
                zip: data.location?.postal,
                lat: data.location?.latitude,
                lon: data.location?.longitude,
                timezone: data.time_zone?.id,
                isp: data.company?.name,
                company_type: data.company?.type,
                security: data.security
            });
        }
    }

    parseAbuseIPDB(data) {
        if (data.ipAddress) {
            this.mergeData({
                abuse_confidence: data.abuseConfidencePercentage,
                country_match: data.countryMatch,
                usage_type: data.usageType,
                is_public: data.isPublic,
                is_whitelisted: data.isWhitelisted,
                total_reports: data.totalReports
            });
        }
    }

    parseFreeIPAPI(data) {
        if (data.ipAddress) {
            this.mergeData({
                country: data.countryName,
                countryCode: data.countryCode,
                region: data.regionName,
                city: data.cityName,
                zip: data.zipCode,
                lat: data.latitude,
                lon: data.longitude,
                timezone: data.timeZone
            });
        }
    }

    // Additional public API parsers
    parseIPAPIIs(data) {
        if (!data) return;
        const city = data.city || data.location?.city || data.geo?.city;
        const region = data.region || data.location?.region || data.subdivision || data.state;
        const country = data.country || data.country_name || data.location?.country;
        const countryCode = data.country_code || data.countryCode || data.location?.country_code;
        const postal = data.postal || data.postal_code || data.zip || data.zip_code;
        const lat = data.latitude ?? data.lat ?? data.location?.latitude ?? data.loc?.lat;
        const lon = data.longitude ?? data.lon ?? data.location?.longitude ?? data.loc?.lng ?? data.loc?.lon;
        const timezone = data.timezone || data.time_zone || data.location?.timezone;
        const isp = data.isp || data.org || data.organization?.name || data.connection?.isp;
        const asn = data.asn || data.as?.asn || data.connection?.asn;
        this.mergeData({ city, region, country, countryCode, postal, zip: postal, lat: parseFloat(lat), lon: parseFloat(lon), timezone, isp, asn });
    }

    parseTechnikNews(data) {
        if (!data) return;
        // TechnikNews IPGeo mirrors ip-api style
        const city = data.city;
        const region = data.regionName || data.region;
        const country = data.country;
        const countryCode = data.countryCode || data.country_code;
        const zip = data.zip || data.postal;
        const lat = data.lat ?? data.latitude;
        const lon = data.lon ?? data.longitude;
        const timezone = data.timezone || data.time_zone;
        const isp = data.isp || data.org;
        const asn = typeof data.as === 'string' ? data.as : data.asn;
        this.mergeData({ city, region, country, countryCode, zip, postal: zip, lat: parseFloat(lat), lon: parseFloat(lon), timezone, isp, asn });
    }

    parseIPLocationNet(data) {
        if (!data) return;
        const city = data.city;
        const region = data.state || data.region;
        const country = data.country_name || data.country;
        const countryCode = data.country_code2 || data.country_code;
        const zip = data.zip_code || data.postal;
        const lat = data.latitude;
        const lon = data.longitude;
        const timezone = data.time_zone || data.timezone;
        const isp = data.isp || data.org || data.organization;
        const asn = data.asn || data.as;
        this.mergeData({ city, region, country, countryCode, zip, postal: zip, lat: parseFloat(lat), lon: parseFloat(lon), timezone, isp, asn });
    }

    parseGeoIPLookup(data) {
        if (!data) return;
        const city = data.city;
        const region = data.region;
        const country = data.country_name || data.country;
        const countryCode = data.country_code;
        const zip = data.postal_code || data.zip_code || data.postal;
        const lat = data.latitude;
        const lon = data.longitude;
        const timezone = data.timezone || data.time_zone;
        const isp = data.isp || data.org || data.organization;
        const asn = data.asn || data.as;
        this.mergeData({ city, region, country, countryCode, zip, postal: zip, lat: parseFloat(lat), lon: parseFloat(lon), timezone, isp, asn });
    }

    parseSypexGeo(data) {
        if (!data) return;
        const city = data.city?.name_en || data.city?.name || data.city?.name_ru;
        const region = data.region?.name_en || data.region?.name || data.region?.name_ru;
        const country = data.country?.name_en || data.country?.name || data.country?.name_ru;
        const countryCode = data.country?.iso || data.country?.iso2 || data.country_code;
        const zip = data.city?.post || data.postal || data.zip;
        const lat = data.city?.lat ?? data.latitude;
        const lon = data.city?.lon ?? data.longitude;
        const timezone = data.timezone || data.region?.timezone || data.country?.timezone;
        this.mergeData({ city, region, country, countryCode, zip, postal: zip, lat: parseFloat(lat), lon: parseFloat(lon), timezone });
    }

    parseGeoPlugin(data) {
        if (!data) return;
        const city = data.geoplugin_city;
        const region = data.geoplugin_region || data.geoplugin_regionName;
        const country = data.geoplugin_countryName;
        const countryCode = data.geoplugin_countryCode;
        const lat = data.geoplugin_latitude;
        const lon = data.geoplugin_longitude;
        // geoPlugin typically doesn't provide postal code reliably
        this.mergeData({ city, region, country, countryCode, lat: parseFloat(lat), lon: parseFloat(lon) });
    }

    mergeData(newData) {
        Object.keys(newData).forEach(key => {
            if (newData[key] !== null && newData[key] !== undefined && newData[key] !== '') {
                if (!this.currentData[key] || this.currentData[key] === 'N/A') {
                    // Sanitize timezone
                    if (key === 'timezone') {
                        const tz = this.normalizeTimezone(newData[key]);
                        if (tz) this.currentData[key] = (tz.type === 'iana') ? tz.value : this.getDisplayTimezone(newData[key]);
                        else this.currentData[key] = 'N/A';
                    } else {
                        this.currentData[key] = newData[key];
                    }
                }
                // Collect aggregates for summary
                const v = newData[key];
                if (key === 'city' && typeof v === 'string') this.aggregates.cities.add(v);
                if (key === 'region' && typeof v === 'string') this.aggregates.regions.add(v);
                // Postal/ZIP code aggregation across different API field names
                const postalKeys = ['postal', 'postal_code', 'postalCode', 'postcode', 'zip', 'zip_code', 'zipCode'];
                if (postalKeys.includes(key)) {
                    if (typeof v === 'string' || typeof v === 'number') {
                        this.aggregates.zips.add(String(v));
                    }
                }
                if ((key === 'country' || key === 'countryName') && typeof v === 'string') {
                    const code = newData.countryCode || newData.country_code || this.currentData.countryCode || this.currentData.country_code || '';
                    const formatted = code ? `${v} (${code})` : v;
                    this.aggregates.countries.add(formatted);
                }
                if ((key === 'isp' || key === 'org' || key === 'organization') && typeof v === 'string') this.aggregates.isps.add(v);
                if ((key === 'as' || key === 'asn') && typeof v === 'string') this.aggregates.asns.add(v);
            }
        });
    }

    async checkMyIP() {
        const myIpv4Element = document.getElementById('myIpv4');
        const myIp = myIpv4Element ? myIpv4Element.textContent : '';
        if (myIp && myIp !== 'Detecting...' && myIp !== 'Not detected' && myIp !== 'Unable to detect' && myIp !== 'Error') {
            document.getElementById('ipInput').value = myIp;
            await this.checkIP();
        } else {
            this.showError('Your IP has not been detected yet. Please wait or enter an IP manually.');
        }
    }

    calculatePrivacyScore() {
        let score = 100;
        const factors = [];

        // Check for proxy/VPN
        if (this.currentData.proxy) {
            score -= 30;
            factors.push('Proxy terdeteksi (-30)');
        }

        // Check for hosting/datacenter
        if (this.currentData.hosting) {
            score -= 25;
            factors.push('Hosting/Datacenter (-25)');
        }

        // Check ISP for VPN patterns
        const isp = (this.currentData.isp || '').toLowerCase();
        const org = (this.currentData.org || '').toLowerCase();
        
        for (const vpnProvider of this.privacyPatterns.vpnProviders) {
            if (isp.includes(vpnProvider.toLowerCase()) || org.includes(vpnProvider.toLowerCase())) {
                score -= 40;
                factors.push(`VPN Provider detected: ${vpnProvider} (-40)`);
                break;
            }
        }

        // Check for cloud providers
        for (const cloudProvider of this.privacyPatterns.cloudProviders) {
            if (isp.includes(cloudProvider.toLowerCase()) || org.includes(cloudProvider.toLowerCase())) {
                score -= 20;
                factors.push(`Cloud Provider: ${cloudProvider} (-20)`);
                break;
            }
        }

        // Check mobile connection
        if (this.currentData.mobile) {
            score += 10;
            factors.push('Mobile connection (+10)');
        }

        // Check abuse reports
        if (this.currentData.abuse_confidence > 0) {
            const penalty = Math.min(50, this.currentData.abuse_confidence);
            score -= penalty;
            factors.push(`Abuse reports (-${penalty})`);
        }

        score = Math.max(0, Math.min(100, score));
        
        this.updatePrivacyMeter(score, factors);
        this.currentData.privacyScore = score;
        this.currentData.privacyFactors = factors;
    }

    updatePrivacyMeter(score, factors) {
        const meter = document.querySelector('.privacy-meter');
        const scoreText = document.querySelector('.score-text');
        const progressCircle = document.querySelector('.privacy-meter .progress');
        
        if (!meter || !scoreText || !progressCircle) return;

        scoreText.textContent = Math.round(score);
        
        // Update circle progress
        const circumference = 2 * Math.PI * 52; // radius = 52
        const offset = circumference - (score / 100) * circumference;
        progressCircle.style.strokeDasharray = circumference;
        progressCircle.style.strokeDashoffset = offset;
        
        // Color based on score
        let color;
        if (score >= 80) color = '#10b981'; // green
        else if (score >= 60) color = '#f59e0b'; // yellow
        else if (score >= 40) color = '#f97316'; // orange
        else color = '#ef4444'; // red
        
        progressCircle.style.stroke = color;
        scoreText.style.color = color;

        // Update factors list
        const factorsList = document.getElementById('privacyFactors');
        if (factorsList && factors.length > 0) {
            factorsList.innerHTML = factors.map(factor => 
                `<li class="privacy-factor">${factor}</li>`
            ).join('');
        }
    }

    updatePrivacyScore() {
        // Initial privacy score display
        const meter = document.querySelector('.privacy-meter');
        if (meter) {
            this.updatePrivacyMeter(0, []);
        }
    }

    displayResults() {
        this.displayBasicInfo();
        this.displaySummaryInfo();
        this.displayLocationInfo();
        this.displayNetworkInfo();
        this.displaySecurityInfo();
        this.displayPrivacyInfo();
        this.setupLazyMap();
        
        // Show results with animation
        const results = document.getElementById('results');
        if (results) {
            results.classList.add('show', 'fade-in');
        }

        // Initialize toggle buttons
        this.initCardToggles();
    }

    displaySummaryInfo() {
        const summary = document.getElementById('summaryInfo');
        if (!summary) return;

        // Prepare lists
        const listify = set => (set.size ? Array.from(set).map(x => `<code>${x}</code>`).join(', ') : 'N/A');
        const countries = listify(this.aggregates.countries);
        const regions = listify(this.aggregates.regions);
        const cities = listify(this.aggregates.cities);
        const zips = listify(this.aggregates.zips);

        // ISP + AS - Show only first result (single entry)
        const ispList = Array.from(this.aggregates.isps);
        const asnList = Array.from(this.aggregates.asns);
        let ispAsCombined = 'N/A';
        if (ispList.length || asnList.length) {
            // Take only the first ISP and first ASN
            const firstIsp = ispList[0] || '';
            const firstAsn = asnList[0] || '';
            const part = [firstIsp, firstAsn].filter(Boolean).join(' • ');
            if (part) {
                ispAsCombined = `<code>${part}</code>`;
            }
        }

        summary.innerHTML = `
            <div class="info-item">
                <span class="info-label"><i class="fas fa-map"></i> Region</span>
                <span class="info-value">${regions}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-flag"></i> Country + ISO</span>
                <span class="info-value">${countries}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-city"></i> City</span>
                <span class="info-value">${cities}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-mail-bulk"></i> ZIP / Postal Code</span>
                <span class="info-value">${zips}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-network-wired"></i> ISP + AS</span>
                <span class="info-value">${ispAsCombined}</span>
            </div>
        `;
    }

    displayBasicInfo() {
        const basicInfo = document.getElementById('basicInfo');
        if (!basicInfo) return;

    const ipType = this.currentData.ip?.includes(':') ? 'IPv6' : 'IPv4';
    const tzLabel = this.getDisplayTimezone(this.currentData.timezone);
    const currentTime = this.getCurrentTimeInTimezone(this.currentData.timezone);

        basicInfo.innerHTML = `
            <div class="info-item">
                <span class="info-label"><i class="fas fa-network-wired"></i> IP Address</span>
                <span class="info-value font-mono">${this.currentData.ip || 'N/A'}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-tag"></i> Type</span>
                <span class="info-value">
                    <span class="status-badge ${ipType === 'IPv6' ? 'info' : 'success'}">${ipType}</span>
                </span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-server"></i> Hostname</span>
                <span class="info-value font-mono">${this.currentData.hostname || this.currentData.reverse || 'N/A'}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-clock"></i> Timezone</span>
                <span class="info-value">${tzLabel}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-calendar-alt"></i> Local Time</span>
                <span class="info-value">${currentTime}</span>
            </div>
        `;
    }

    displayLocationInfo() {
        const locationInfo = document.getElementById('locationInfo');
        if (!locationInfo) return;

        locationInfo.innerHTML = `
            <div class="info-item">
                <span class="info-label"><i class="fas fa-globe-americas"></i> Continent</span>
                <span class="info-value">${this.currentData.continent || 'N/A'}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-flag"></i> Country</span>
                <span class="info-value">${this.currentData.country || 'N/A'} ${this.currentData.countryCode ? `(${this.currentData.countryCode})` : ''}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-map"></i> Region</span>
                <span class="info-value">${this.currentData.region || 'N/A'}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-city"></i> City</span>
                <span class="info-value">${this.currentData.city || 'N/A'}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-building"></i> District</span>
                <span class="info-value">${this.currentData.district || 'N/A'}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-mail-bulk"></i> Postal Code</span>
                <span class="info-value">${this.currentData.zip || this.currentData.postal || 'N/A'}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-crosshairs"></i> Coordinates</span>
                <span class="info-value font-mono">${this.currentData.lat && this.currentData.lon ? `${this.currentData.lat}, ${this.currentData.lon}` : 'N/A'}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-money-bill"></i> Currency</span>
                <span class="info-value">${this.currentData.currency || 'N/A'}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-language"></i> Languages</span>
                <span class="info-value">${this.currentData.languages || 'N/A'}</span>
            </div>
        `;
    }

    displayNetworkInfo() {
        const networkInfo = document.getElementById('networkInfo');
        if (!networkInfo) return;

        networkInfo.innerHTML = `
            <div class="info-item">
                <span class="info-label"><i class="fas fa-wifi"></i> ISP</span>
                <span class="info-value">${this.currentData.isp || 'N/A'}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-building"></i> Organization</span>
                <span class="info-value">${this.currentData.org || 'N/A'}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-hashtag"></i> AS Number</span>
                <span class="info-value font-mono">${this.currentData.as || this.currentData.asn || 'N/A'}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-tag"></i> AS Name</span>
                <span class="info-value">${this.currentData.asname || 'N/A'}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-phone"></i> Calling Code</span>
                <span class="info-value">${this.currentData.calling_code || 'N/A'}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-globe"></i> Country TLD</span>
                <span class="info-value">${this.currentData.country_tld || 'N/A'}</span>
            </div>
        `;
    }

    displaySecurityInfo() {
        const securityInfo = document.getElementById('securityInfo');
        if (!securityInfo) return;

        const abuseLevel = this.getAbuseLevel(this.currentData.abuse_confidence || 0);
        const threatLevel = this.getThreatLevel();

        securityInfo.innerHTML = `
            <div class="info-item">
                <span class="info-label"><i class="fas fa-mobile-alt"></i> Mobile Connection</span>
                <span class="info-value">
                    <span class="status-badge ${this.currentData.mobile ? 'success' : 'info'}">
                        ${this.currentData.mobile ? 'Yes' : 'No'}
                    </span>
                </span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-user-secret"></i> Proxy/VPN</span>
                <span class="info-value">
                    <span class="status-badge ${this.currentData.proxy ? 'warning' : 'success'}">
                        ${this.currentData.proxy ? 'Detected' : 'Not Detected'}
                    </span>
                </span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-server"></i> Hosting/Datacenter</span>
                <span class="info-value">
                    <span class="status-badge ${this.currentData.hosting ? 'warning' : 'success'}">
                        ${this.currentData.hosting ? 'Yes' : 'No'}
                    </span>
                </span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-shield-alt"></i> Abuse Confidence</span>
                <span class="info-value">
                    <span class="status-badge ${abuseLevel.class}">
                        ${this.currentData.abuse_confidence || 0}% ${abuseLevel.text}
                    </span>
                </span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-exclamation-triangle"></i> Threat Level</span>
                <span class="info-value">
                    <span class="status-badge ${threatLevel.class}">
                        ${threatLevel.text}
                    </span>
                </span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-eye"></i> Usage Type</span>
                <span class="info-value">${this.currentData.usage_type || 'Unknown'}</span>
            </div>
            <div class="info-item">
                <span class="info-label"><i class="fas fa-globe-americas"></i> Public IP</span>
                <span class="info-value">
                    <span class="status-badge ${this.currentData.is_public ? 'info' : 'warning'}">
                        ${this.currentData.is_public ? 'Yes' : 'No/Unknown'}
                    </span>
                </span>
            </div>
        `;
    }

    displayPrivacyInfo() {
        const privacyInfo = document.getElementById('privacyInfo');
        if (!privacyInfo) return;

        privacyInfo.innerHTML = `
            <div class="privacy-score">
                <div class="privacy-meter">
                    <svg width="120" height="120">
                        <circle class="background" cx="60" cy="60" r="52"></circle>
                        <circle class="progress" cx="60" cy="60" r="52"></circle>
                    </svg>
                    <div class="score-text">0</div>
                </div>
                <h4>Privacy Score</h4>
                <p class="text-sm text-gray-600">Score based on privacy and security analysis</p>
                <ul id="privacyFactors" class="privacy-factors"></ul>
            </div>
        `;
    }

    getAbuseLevel(confidence) {
        if (confidence >= 75) return { class: 'danger', text: 'High Risk' };
        if (confidence >= 50) return { class: 'warning', text: 'Medium Risk' };
        if (confidence >= 25) return { class: 'info', text: 'Low Risk' };
        return { class: 'success', text: 'Clean' };
    }

    getThreatLevel() {
        let threatScore = 0;
        
        if (this.currentData.proxy) threatScore += 30;
        if (this.currentData.hosting) threatScore += 20;
        if (this.currentData.abuse_confidence > 50) threatScore += 40;
        if (this.currentData.total_reports > 10) threatScore += 20;

        if (threatScore >= 70) return { class: 'danger', text: 'High' };
        if (threatScore >= 40) return { class: 'warning', text: 'Medium' };
        if (threatScore >= 20) return { class: 'info', text: 'Low' };
        return { class: 'success', text: 'Minimal' };
    }

    getCurrentTimeInTimezone(timezone) {
        const tz = this.normalizeTimezone(timezone);
        if (!tz) return 'N/A';

        const now = new Date();
        // If tz is IANA name
        if (tz.type === 'iana') {
            try {
                return new Intl.DateTimeFormat('en-GB', {
                    timeZone: tz.value,
                    year: 'numeric',
                    month: '2-digit',
                    day: '2-digit',
                    hour: '2-digit',
                    minute: '2-digit',
                    second: '2-digit',
                    hour12: false
                }).format(now);
            } catch {
                // fallthrough to offset formatting if needed
            }
        }
        // If tz is offset in minutes
        if (tz.type === 'offset') {
            const local = new Date(now.getTime() + (tz.minutes - now.getTimezoneOffset()) * 60000);
            return new Intl.DateTimeFormat('en-GB', {
                year: 'numeric', month: '2-digit', day: '2-digit',
                hour: '2-digit', minute: '2-digit', second: '2-digit', hour12: false
            }).format(local);
        }
        return 'N/A';
    }

    getDisplayTimezone(timezone) {
        const tz = this.normalizeTimezone(timezone);
        if (!tz) return 'N/A';
        if (tz.type === 'iana') return tz.value;
        if (tz.type === 'offset') {
            const sign = tz.minutes >= 0 ? '+' : '-';
            const abs = Math.abs(tz.minutes);
            const hh = String(Math.floor(abs / 60)).padStart(2, '0');
            const mm = String(abs % 60).padStart(2, '0');
            return `UTC${sign}${hh}:${mm}`;
        }
        return 'N/A';
    }

    normalizeTimezone(timezone) {
        if (!timezone) return null;
        // If object, try common properties
        if (typeof timezone === 'object') {
            // Some APIs return { id: 'Region/City' } or { name: 'Region/City' } or { offset: -180 }
            const id = timezone.id || timezone.name || timezone.timezone || timezone.tz;
            if (id && typeof id === 'string') return { type: 'iana', value: id };
            const offset = timezone.offset_minutes || timezone.offsetMinutes || timezone.gmtOffset || timezone.offset;
            if (typeof offset === 'number') return { type: 'offset', minutes: offset };
            // Some return seconds
            const offsetSec = timezone.offset_seconds || timezone.offsetSeconds;
            if (typeof offsetSec === 'number') return { type: 'offset', minutes: Math.round(offsetSec / 60) };
            return null;
        }
        if (typeof timezone === 'string') {
            const tzStr = timezone.trim();
            // If looks like IANA "Area/City"
            if (/^[A-Za-z]+\/[A-Za-z_+-]+/.test(tzStr)) return { type: 'iana', value: tzStr };
            // If looks like UTC offset: "+02:00", "-0700", "+2", "UTC+7", "GMT-03:30"
            const m = tzStr.match(/([+-]?)(\d{1,2})(?::?(\d{2}))?/);
            if (m) {
                const sign = m[1] === '-' ? -1 : 1;
                const h = parseInt(m[2], 10);
                const mm = m[3] ? parseInt(m[3], 10) : 0;
                if (!isNaN(h) && h <= 14) {
                    return { type: 'offset', minutes: sign * (h * 60 + mm) };
                }
            }
            // Common aliases
            if (/^(utc|gmt)$/i.test(tzStr)) return { type: 'offset', minutes: 0 };
            // Fallback: return as IANA and let formatter try; display code will handle failure
            return { type: 'iana', value: tzStr };
        }
        return null;
    }

    setupLazyMap() {
        if (!this.currentData.lat || !this.currentData.lon) return;

        const mapContainer = document.getElementById('map');
        if (!mapContainer) return;

        // Add loading placeholder
        mapContainer.innerHTML = `
            <div class="map-lazy-placeholder">
                <div class="map-lazy-content">
                    <i class="fas fa-map-marked-alt" style="font-size: 3rem; color: var(--accent-green); margin-bottom: 1rem;"></i>
                    <h3 style="margin: 0 0 0.5rem 0; color: var(--text-primary);">Interactive Map</h3>
                    <p style="margin: 0 0 1rem 0; color: var(--gray-600);">
                        ${this.currentData.city || 'Unknown'}, ${this.currentData.country || 'Unknown'}
                    </p>
                    <button class="btn btn-primary btn-load-map">
                        <i class="fas fa-eye"></i> Load Map
                    </button>
                </div>
            </div>
        `;

        // Add event listener to load button
        const loadButton = mapContainer.querySelector('.btn-load-map');
        if (loadButton) {
            loadButton.addEventListener('click', () => {
                this.initMap();
            });
        }

        // Optional: Auto-load after delay or on scroll
        if (window.IP_CHECKER_CONFIG?.MAP_LAZY_LOAD) {
            // Auto-load after 3 seconds
            setTimeout(() => {
                if (mapContainer.querySelector('.map-lazy-placeholder')) {
                    this.initMap();
                }
            }, 3000);
        }
    }

    initMap() {
        if (!this.currentData.lat || !this.currentData.lon) return;

        const mapContainer = document.getElementById('map');
        if (!mapContainer) return;

        // Show loading state
        mapContainer.innerHTML = `
            <div class="map-loading">
                <div class="loading-box">
                    <div class="spinner"></div>
                    <p class="loading-text">Loading map...</p>
                </div>
            </div>
        `;

        // Remove existing map
        if (this.map) {
            this.map.remove();
        }

        // Short delay to show loading
        setTimeout(() => {
            // Clear container
            mapContainer.innerHTML = '';

            // Initialize map
            this.map = L.map('map').setView([this.currentData.lat, this.currentData.lon], 10);

            // Ensure Leaflet recalculates size after being displayed
            setTimeout(() => { if (this.map) this.map.invalidateSize(); }, 50);

            // Add tile layer with loading optimization
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                attribution: '© OpenStreetMap contributors',
                maxZoom: 18,
                detectRetina: true,
                updateWhenZooming: false,
                updateWhenIdle: true
            }).addTo(this.map);

            // Create custom icon
            const customIcon = L.divIcon({
                className: 'custom-map-marker',
                html: '<i class="fas fa-map-marker-alt" style="color: var(--accent-green); font-size: 2rem;"></i>',
                iconSize: [30, 30],
                iconAnchor: [15, 30]
            });

            // Add marker
            const marker = L.marker([this.currentData.lat, this.currentData.lon], {
                icon: customIcon
            }).addTo(this.map);

            // Popup content
            const popupContent = `
                <div class="map-popup">
                    <strong>${this.currentData.city || 'Unknown City'}</strong><br>
                    ${this.currentData.region || 'Unknown Region'}, ${this.currentData.country || 'Unknown Country'}<br>
                    <strong>IP:</strong> ${this.currentData.ip}<br>
                    <strong>ISP:</strong> ${this.currentData.isp || 'Unknown'}<br>
                    <strong>Coordinates:</strong> ${this.currentData.lat}, ${this.currentData.lon}
                </div>
            `;

            marker.bindPopup(popupContent).openPopup();

            // Add circle to show approximate area (theme green)
            L.circle([this.currentData.lat, this.currentData.lon], {
                color: '#00ff88',
                fillColor: '#00ff88',
                fillOpacity: 0.08,
                radius: 5000 // 5km radius
            }).addTo(this.map);
        }, 500); // Small delay for loading effect

    }

    showAPIStatus() {
        const apiStatus = document.getElementById('apiStatus');
        const apiStatusList = document.getElementById('apiStatusList');
        
        if (!apiStatus || !apiStatusList) return;

        let html = '';
        Object.entries(this.apiResults).forEach(([apiKey, result]) => {
            const api = this.apis[apiKey];
            if (!api) return;

            const statusClass = result.status === 'success' ? 'success' : 'error';
            const statusText = result.status === 'success' ? 'Success' : 'Failed';
            
            html += `
                <div class="api-item">
                    <div class="api-info">
                        <span class="api-name">${api.name}</span>
                        <span class="api-response-time">${result.responseTime}ms</span>
                    </div>
                    <div class="api-status-indicator ${statusClass}">
                        ${statusText}
                    </div>
                </div>
            `;
        });
        
        apiStatusList.innerHTML = html;
        apiStatus.classList.add('show');

        // Re-init toggles in case this section was collapsed
        this.initCardToggles();
    }

    initCardToggles() {
        const buttons = document.querySelectorAll('.toggle-btn');
        buttons.forEach(btn => {
            if (btn.dataset.bound) return; // prevent duplicate bindings
            btn.dataset.bound = '1';
            btn.addEventListener('click', () => {
                const targetSel = btn.getAttribute('data-target');
                const card = document.querySelector(targetSel);
                if (!card) return;
                const body = card.querySelector('.card-body');
                const collapsed = card.classList.toggle('collapsed');
                if (body) body.style.display = collapsed ? 'none' : '';
                btn.textContent = collapsed ? 'Show' : 'Hide';
            });
        });
    }

    updateProgress(percentage) {
        const progressBar = document.querySelector('.loading-progress-bar');
        if (progressBar) {
            progressBar.style.width = `${percentage}%`;
        }
    }

    showLoading() {
        const loading = document.getElementById('loading');
        const results = document.getElementById('results');
        const apiStatus = document.getElementById('apiStatus');
        
        if (loading) loading.classList.add('show');
        if (results) results.classList.remove('show');
        if (apiStatus) apiStatus.classList.remove('show');
    }

    hideLoading() {
        const loading = document.getElementById('loading');
        if (loading) loading.classList.remove('show');
    }

    showError(message) {
        const errorDiv = document.getElementById('errorMessage');
        if (errorDiv) {
            errorDiv.innerHTML = `<i class="fas fa-exclamation-triangle"></i> ${message}`;
            errorDiv.classList.add('show');
        }
    }

    clearError() {
        const errorDiv = document.getElementById('errorMessage');
        if (errorDiv) {
            errorDiv.classList.remove('show');
        }
    }

    clearResults() {
        const ipInput = document.getElementById('ipInput');
        const results = document.getElementById('results');
        const apiStatus = document.getElementById('apiStatus');
        
        if (ipInput) {
            ipInput.value = '';
            ipInput.classList.remove('valid', 'invalid');
        }
        if (results) results.classList.remove('show');
        if (apiStatus) apiStatus.classList.remove('show');
        
        this.clearError();
        this.hideLoading();
        this.currentData = {};
        this.apiResults = {};
        
        if (this.map) {
            this.map.remove();
            this.map = null;
        }
        
        this.updatePrivacyScore();
    }

    async handleConvert() {
        const input = document.getElementById('converterInput');
        const resultBox = document.getElementById('converterResult');
        if (!input || !resultBox) return;

        const value = (input.value || '').trim();
        
        if (!value) {
            resultBox.innerHTML = `<div class="info-item"><span class="info-label">Result</span><span class="info-value">Enter an IPv6 address first.</span></div>`;
            return;
        }

        const validation = IPUtils.validateIP(value);
        if (!validation.valid || validation.type !== 'IPv6') {
            resultBox.innerHTML = `<div class="info-item"><span class="info-label">Result</span><span class="info-value">Input must be a valid IPv6 address.</span></div>`;
            return;
        }

        // Show loading
        resultBox.innerHTML = `
            <div class="loading-box">
                <div class="spinner"></div>
                <p class="loading-text">Detecting dual-stack IPs...</p>
            </div>
        `;

        try {
            const dualStackInfo = await this.detectDualStack(value);
            this.displayDualStackResults(dualStackInfo, resultBox);
        } catch (error) {
            console.error('Dual-stack detection error:', error);
            resultBox.innerHTML = `<div class="info-item"><span class="info-label">Error</span><span class="info-value">Failed to detect dual-stack information.</span></div>`;
        }
    }

    async detectDualStack(ipv6Address) {
        const results = {
            inputIPv6: ipv6Address,
            foundIPv4: null,
            detectionMethods: [],
            apiResults: {},
            confidence: 'Unknown'
        };

        // Method 1: Try IPv4-mapped IPv6 conversion
        const conv = IPUtils.ipv6ToIpv4(ipv6Address);
        if (conv.success) {
            results.foundIPv4 = conv.ipv4;
            results.detectionMethods.push({
                method: conv.method,
                result: conv.ipv4,
                confidence: 'High'
            });
        }

        // Method 2: Query target IPv6 through geolocation APIs
        const targetAnalysisAPIs = [
            {
                name: 'IPInfo Target Analysis',
                url: `https://ipinfo.io/${ipv6Address}/json`,
                parser: 'parseIPInfo'
            },
            {
                name: 'IPWho.is IPv6 Analysis', 
                url: `https://ipwho.is/${ipv6Address}`,
                parser: 'parseIPWhoIs'
            },
            {
                name: 'IPAPI.co IPv6 Analysis',
                url: `https://ipapi.co/${ipv6Address}/json/`,
                parser: 'parseIPAPICo'
            },
            {
                name: 'FreeGeoIP IPv6 Analysis',
                url: `https://freegeoip.app/json/${ipv6Address}`,
                parser: 'parseFreeGeoIP'
            },
            {
                name: 'GeoJS IPv6 Analysis',
                url: `https://get.geojs.io/v1/ip/geo/${ipv6Address}.json`,
                parser: 'parseGeoJS'
            }
        ];

        // Fetch geolocation data for the target IPv6
        const promises = targetAnalysisAPIs.map(async (api) => {
            try {
                const response = await fetch(api.url, { 
                    method: 'GET',
                    signal: AbortSignal.timeout(7000)
                });
                
                if (!response.ok) throw new Error(`HTTP ${response.status}`);
                
                const data = await response.json();
                
                results.apiResults[api.name] = {
                    success: true,
                    data: data,
                    location: `${data.city || 'Unknown'}, ${data.country || 'Unknown'}`,
                    isp: data.org || data.isp || data.as || 'Unknown'
                };
                
                return { api, data };
            } catch (error) {
                results.apiResults[api.name] = {
                    success: false,
                    error: error.message
                };
                return null;
            }
        });

        const apiResponses = await Promise.all(promises);

        // Method 3: Try to reverse lookup or find associated IPv4 networks
        // Many times, the same ISP/organization will have both IPv4 and IPv6 blocks
        let organizationInfo = null;
        let locationInfo = null;
        
        apiResponses.forEach(response => {
            if (response && response.data) {
                const data = response.data;
                
                // Collect organization/ISP information
                if (!organizationInfo && (data.org || data.isp || data.as)) {
                    organizationInfo = data.org || data.isp || data.as;
                }
                
                // Collect location information 
                if (!locationInfo && data.city && data.country) {
                    locationInfo = {
                        city: data.city,
                        country: data.country,
                        region: data.region || data.regionName
                    };
                }
            }
        });

        // Method 4: Try to find IPv4 ranges from the same organization
        if (organizationInfo) {
            try {
                // Search for known IPv4 blocks from same ISP/organization
                const orgSearchResult = await this.searchIPv4ByOrganization(organizationInfo, locationInfo);
                if (orgSearchResult && orgSearchResult.potentialIPv4) {
                    results.foundIPv4 = orgSearchResult.potentialIPv4;
                    results.detectionMethods.push({
                        method: 'Organization Network Analysis',
                        result: orgSearchResult.potentialIPv4,
                        confidence: 'Medium',
                        details: `Found IPv4 from same ISP: ${organizationInfo}`
                    });
                }
            } catch (error) {
                console.warn('Organization search failed:', error);
            }
        }

        // Method 5: DNS reverse lookup attempt
        try {
            const dnsResult = await this.tryDNSReverseLookup(ipv6Address);
            if (dnsResult && dnsResult.hostname) {
                results.detectionMethods.push({
                    method: 'DNS Reverse Lookup',
                    result: dnsResult.hostname,
                    confidence: 'Low',
                    details: 'Found hostname, IPv4 may be resolvable from same domain'
                });
                
                // Try to resolve IPv4 from the same hostname
                const ipv4FromDNS = await this.resolveIPv4FromHostname(dnsResult.hostname);
                if (ipv4FromDNS) {
                    results.foundIPv4 = ipv4FromDNS;
                    results.detectionMethods[results.detectionMethods.length - 1].confidence = 'High';
                    results.detectionMethods[results.detectionMethods.length - 1].result = ipv4FromDNS;
                }
            }
        } catch (error) {
            console.warn('DNS lookup failed:', error);
        }

        // Set overall confidence based on detection methods
        if (results.detectionMethods.length > 0) {
            const hasHighConfidence = results.detectionMethods.some(m => m.confidence === 'High');
            const hasMediumConfidence = results.detectionMethods.some(m => m.confidence === 'Medium');
            
            if (hasHighConfidence) {
                results.confidence = 'High';
            } else if (hasMediumConfidence) {
                results.confidence = 'Medium';
            } else {
                results.confidence = 'Low';
            }
        }

        // Store analysis metadata
        results.organizationInfo = organizationInfo;
        results.locationInfo = locationInfo;

        return results;
    }

    async searchIPv4ByOrganization(organization, location) {
        // This is a simplified approach - in real world, you'd need a comprehensive IP database
        // For now, we'll use a heuristic approach based on common ISP patterns
        
        try {
            // Try to query known IPv4 ranges from same organization
            const response = await fetch(`https://ipinfo.io/8.8.8.8/json`, {
                signal: AbortSignal.timeout(3000)
            });
            
            if (response.ok) {
                const data = await response.json();
                // This is a placeholder - real implementation would query IP range databases
                return null; // No reliable method without dedicated databases
            }
        } catch (error) {
            console.warn('Organization search error:', error);
        }
        
        return null;
    }

    async tryDNSReverseLookup(ipv6Address) {
        // Note: Browser JavaScript cannot directly perform DNS lookups
        // This is a placeholder for potential future API integration
        try {
            // You could integrate with DNS-over-HTTPS services here
            // For now, we'll skip this method in browser environment
            return null;
        } catch (error) {
            return null;
        }
    }

    async resolveIPv4FromHostname(hostname) {
        // Browser limitation - cannot directly resolve DNS
        // Would need a backend service or DNS-over-HTTPS integration
        return null;
    }

    parseCloudflareTrace(text) {
        const lines = text.split('\n');
        const data = {};
        lines.forEach(line => {
            const [key, value] = line.split('=');
            if (key && value) {
                data[key.trim()] = value.trim();
            }
        });
        return data;
    }

    displayDualStackResults(results, resultBox) {
        const lines = [];
        
        // Header with target IPv6
        lines.push(`<div class="info-item"><span class="info-label">Target IPv6</span><span class="info-value font-mono">${results.inputIPv6}</span></div>`);
        
        // Show organization/ISP information if found
        if (results.organizationInfo) {
            lines.push(`<div class="info-item"><span class="info-label">Organization</span><span class="info-value">${results.organizationInfo}</span></div>`);
        }
        
        // Show location information if found
        if (results.locationInfo) {
            const location = `${results.locationInfo.city}, ${results.locationInfo.country}`;
            lines.push(`<div class="info-item"><span class="info-label">Location</span><span class="info-value">${location}</span></div>`);
        }
        
        if (results.foundIPv4) {
            lines.push(`<div class="info-item"><span class="info-label">Associated IPv4</span><span class="info-value font-mono" style="color: var(--accent-green);">${results.foundIPv4}</span></div>`);
            lines.push(`<div class="info-item"><span class="info-label">Confidence</span><span class="info-value">${results.confidence}</span></div>`);
            
            // Show detection methods
            if (results.detectionMethods.length > 0) {
                lines.push(`<div class="info-item"><span class="info-label">Detection Methods</span><span class="info-value">${results.detectionMethods.length} method(s) used</span></div>`);
                results.detectionMethods.forEach((method, idx) => {
                    lines.push(`<div class="info-item"><span class="info-label">Method ${idx + 1}</span><span class="info-value">${method.method} (${method.confidence})</span></div>`);
                    if (method.details) {
                        lines.push(`<div class="info-item"><span class="info-label">Details</span><span class="info-value" style="font-size: 0.9em; color: var(--text-secondary);">${method.details}</span></div>`);
                    }
                });
            }
        } else {
            lines.push(`<div class="info-item"><span class="info-label">Associated IPv4</span><span class="info-value">Not detected</span></div>`);
            
            // Provide more specific explanation based on what we found
            if (results.organizationInfo) {
                lines.push(`<div class="info-item"><span class="info-label">Analysis</span><span class="info-value">Found organization: ${results.organizationInfo}</span></div>`);
                lines.push(`<div class="info-item"><span class="info-label">Note</span><span class="info-value">This IPv6 belongs to a known ISP, but no associated IPv4 was detected through available methods.</span></div>`);
            } else {
                lines.push(`<div class="info-item"><span class="info-label">Note</span><span class="info-value">This IPv6 may be:</span></div>`);
                lines.push(`<div class="info-item"><span class="info-label"></span><span class="info-value">• Pure IPv6-only network</span></div>`);
                lines.push(`<div class="info-item"><span class="info-label"></span><span class="info-value">• Using different network infrastructure</span></div>`);
                lines.push(`<div class="info-item"><span class="info-label"></span><span class="info-value">• Behind NAT66 or other IPv6 translation</span></div>`);
            }
        }

        // Show API response summary
        const successfulAPIs = Object.values(results.apiResults).filter(r => r.success).length;
        const totalAPIs = Object.keys(results.apiResults).length;
        lines.push(`<div class="info-item"><span class="info-label">API Status</span><span class="info-value">${successfulAPIs}/${totalAPIs} APIs responded successfully</span></div>`);
        
        // Show available analysis data
        const successfulResults = Object.entries(results.apiResults).filter(([name, result]) => result.success);
        if (successfulResults.length > 0) {
            lines.push(`<div class="info-item"><span class="info-label">Available Data</span><span class="info-value">Geolocation, ISP, and network information collected</span></div>`);
        }

        resultBox.innerHTML = lines.join('');
    }

    exportResults() {
        if (!this.currentData.ip) {
            this.showError('No data to export. Please analyze an IP first.');
            return;
        }

        const exportData = {
            timestamp: new Date().toISOString(),
            ip_info: this.currentData,
            api_results: this.apiResults
        };

        // Create and download JSON file
        const blob = new Blob([JSON.stringify(exportData, null, 2)], { type: 'application/json' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `ip-info-${this.currentData.ip}-${Date.now()}.json`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
    }

    processAPIResults(results) {
        results.forEach((result, index) => {
            if (result.status === 'fulfilled' && result.value) {
                // Already processed in callAPIWithRetry
            } else if (result.status === 'rejected') {
                console.warn('API call failed:', result.reason);
            }
        });
    }
}

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.ipChecker = new IPChecker();
});

// Export for global access
if (typeof module !== 'undefined' && module.exports) {
    module.exports = IPChecker;
}